"""Module contains functions used to organize files."""
from shutil import rmtree, copy, move
from os import path, walk
from pathlib import Path
from re import match
from zipfile import ZipFile, ZIP_DEFLATED


def create_directory(directory_name):
    """
    Creates a new directory at a path described by the directory name string.

    Args:
        (str) directory_name:

    Returns:
        (path) path to the new directory.
    """
    new_directory_path = Path(directory_name)
    new_directory_path.mkdir(parents=True)
    print(f'Directory "{new_directory_path}" created.')
    return new_directory_path


def delete_directory(directory_reference):
    """
    Deletes a directory at a path identified by the directory handle.

    Note that this function executes successfully even when the requested directory does not exist.

    Args:
        directory_reference: Can be either a Path object or a str object.

    Returns:
        None.
    """
    if isinstance(directory_reference, Path):
        directory_path = directory_reference
    elif isinstance(directory_reference, str):
        directory_path = Path(directory_reference)
    else:
        raise TypeError('Directory reference must be a Path or str object.')
    if path.exists(directory_path):
        rmtree(directory_path)
        print(f'Directory "{directory_path}" deleted.')
    else:
        print(f'Directory "{directory_path}" does not exist.')


def show_directory(directory_reference):
    """
    Shows contents of a directory at a path identified by the directory handle.

    Args:
        directory_reference: Can be either a Path object or a str object.

    Returns:
        None.
    """
    directory_path = get_directory_path(directory_reference)
    directory_count = 0
    print(f'\nSub-directories in directory "{directory_path}":')
    for item in sorted(directory_path.iterdir()):
        if item.is_dir():
            directory_count += 1
            print(f'\t{item.name}')
    print(f'\t[There are {directory_count} sub-directories in the directory.]')

    file_count = 0
    print(f'\nFiles in directory "{directory_path}":')
    for item in sorted(directory_path.iterdir()):
        if item.is_file():
            file_count += 1
            print(f'\t{item.name}')
    print(f'\t[There are {file_count} files in the directory.]')


def copy_file(file_reference, directory_reference):
    """
    Copies a file from a path identified by the file reference to a directory identified by the directory reference.

    Note that the file will overwrite a file in the target directory with the same name.

    Args:
        file_reference:        Can be either a Path object or a str object.
        directory_reference:   Can be either a Path object or a str object.

    Returns:
        (path) path to copied file.
    """
    file_path = get_file_path(file_reference)
    directory_path = get_directory_path(directory_reference)
    destination_file = path.join(directory_path, path.basename(file_path))
    copy(file_path, destination_file)
    print(f'Copied "{file_path}" to "{directory_path}".')
    return Path(destination_file)


def rename_file(file_reference, new_name):
    """
    Renames a file at a path identified by the file reference to a new name.

    Args:
        file_reference: Can be either a Path object or a str object.
        new_name:       (str) filename.

    Returns:
        (path) path to the renamed file.

    """
    file_path = get_file_path(file_reference)
    name_parts = new_name.split('.')
    if len(name_parts) != 2:
        raise TypeError(f'New name "{new_name}" is not a valid name.')
    base_name = name_parts[0]
    extension = name_parts[1]
    if extension not in ['csv', 'xml', 'jp2', 'md', 'txt', 'py', 'ipynb']:
        raise ValueError(f'New name "{new_name}" does not have a supported file extension.')
    if not match(r"^[a-zA-Z0-9_\-\s]+$", base_name):
        raise ValueError(f'Base name portion of "{new_name}" contains unsupported characters.')

    new_file_path = file_path.parent / new_name
    file_path.rename(new_file_path)
    print(f'Renamed "{file_path}" to "{new_file_path}".')
    return new_file_path


def clone_file(file_reference, directory_reference, new_name):
    """
    Copies a file to a new directory and renames the new copy.

    Args:
        file_reference:        Can be either a Path object or a str object.
        directory_reference:   Can be either a Path object or a str object.
        new_name:              (str) new filename.

    Returns:
        (path) path to cloned file.

    """
    new_file_path = copy_file(file_reference, directory_reference)
    renamed_file_path = rename_file(new_file_path, new_name)
    print('Clone is complete.')
    return renamed_file_path


def move_file(file_reference, directory_reference):
    """
    Moves a file to a new directory.

    Args:
        file_reference:        Can be either a Path object or a str object.
        directory_reference:   Can be either a Path object or a str object.

    Returns:
        None.
    """
    file_path = get_file_path(file_reference)
    directory_path = get_directory_path(directory_reference)
    move(file_path, directory_path)
    print(f'Moved "{file_path}" to "{directory_path}".')


def delete_file(file_reference):
    """
    Deletes a file.

    Args:
        file_reference:  Can be either a Path object or a str object.

    Returns:
        None.
    """
    if isinstance(file_reference, Path):
        file_path = file_reference
    elif isinstance(file_reference, str):
        file_path = Path(file_reference)
    else:
        raise TypeError('File reference must be a Path or str object.')
    if file_path.is_file():
        file_path.unlink()
        print(f'Deleted "{file_path}".')
    else:
        print(f'File "{file_path}" does not exist.')


def unzip_file(zipfile_reference, directory_reference):
    """
    Unzips a zip file to a specified directory.

    Args:
        zipfile_reference:     Can be either a Path object or a str object.
        directory_reference:   Can be either a Path object or a str object.

    Raises:
        TypeError: File reference is not a zip file.

    Returns:
        None.
    """
    zipfile_path = get_file_path(zipfile_reference)
    if not zipfile_path.suffix == '.zip':
        raise TypeError(f'File reference "{zipfile_path}" is not a zip file.')
    directory_path = get_directory_path(directory_reference)
    with ZipFile(zipfile_path, 'r') as zip_file:
        zip_file.extractall(directory_path)
    print(f'Unzipped "{zipfile_path}" to "{directory_path}".')


def create_zipfile(source_directory_reference, destination_directory_reference, zipfile_name):
    """
    Creates a zip file from a source directory,a destination directory, and a zipfile name.

    Args:
        source_directory_reference:        Can be either a Path object or a str object.
        destination_directory_reference:   Can be either a Path object or a str object.
        zipfile_name:                      (str) zip file name.

    Raises:
        TypeError:  Zipfile name is not a valid name.
        ValueError: Zipfile name does not have a .zip extension.

    Returns:
        None.
    """
    source_directory_path = get_directory_path(source_directory_reference)
    destination_directory_path = get_directory_path(destination_directory_reference)
    name_parts = zipfile_name.split('.')
    if len(name_parts) != 2:
        raise TypeError(f'Zipfile name "{zipfile_name}" is not a valid name.')
    extension = name_parts[1]
    if extension != 'zip':
        raise ValueError(f'New name "{zipfile_name}" does not have a .zip file extension.')
    output_file_path = destination_directory_path / zipfile_name
    with ZipFile(output_file_path, 'w', ZIP_DEFLATED) as output_zipfile:
        # attribution: the directory walking code below was generated by the Google AI bot
        for root, dirs, files in walk(source_directory_path):
            for file in files:
                file_path = path.join(root, file)
                # Calculate the relative path within the zip file
                # This ensures the directory structure inside the zip matches the source
                arcname = path.relpath(file_path, source_directory_path)
                output_zipfile.write(file_path, arcname)
    print(f"Directory '{source_directory_path}' successfully zipped to '{output_file_path}'")


def get_directory_path(directory_reference):
    """
    Get a directory path for a directory reference.

    Args:
        directory_reference: Can be either a Path object or a str object.

    Returns:
        (path) Directory path.

    Raises:
        TypeError: Directory reference must be a Path or str object.
        NotADirectoryError: Directory reference is not a directory.
    """
    if isinstance(directory_reference, Path):
        directory_path = directory_reference
    elif isinstance(directory_reference, str):
        directory_path = Path(directory_reference)
    else:
        raise TypeError('Directory reference must be a Path or str object.')
    if not directory_path.is_dir():
        raise NotADirectoryError(f'Directory reference "{directory_path}" is not a directory.')
    return directory_path


def get_file_path(file_reference):
    """
    Get a file path for a file reference.

    Args:
        file_reference: Can be either a Path object or a str object.

    Returns:
        (path) File path.

    Raises:
        TypeError: File reference must be a Path or str object.
        TypeError: File reference is not a file.
    """
    if isinstance(file_reference, Path):
        file_path = file_reference
    elif isinstance(file_reference, str):
        file_path = Path(file_reference)
    else:
        raise TypeError('File reference must be a Path or str object.')
    if not file_path.is_file():
        raise TypeError(f'File reference "{file_path}" is not a file.')
    return file_path
